﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.IO;

#if M9330
using Agilent.AgM933x.Interop;
#endif
using Ivi.Visa.Interop;

namespace PowerAmpDemoNS
{
    public class Arb
    {
        // ET Arb Variables
        public bool useEtArb = false;
        public bool useM9330 = false;
        public bool use33522B = false;
        public string arbResource = "PXI19::15::0::INSTR";
        public double arbTrigDelay = 0;
#if M9330
        public IAgM933x M9330;
#endif
        public int arb;
        FormattedIO488 arbIo;

        ResourceManager oRm;

        public ArbInfo[] etArbFiles;
        private ArbInfo lastArb = new ArbInfo();
        public void initEtArb()
        {
            if (!useEtArb)
                return;
            if (etArbFiles == null)
                etArbFiles = new ArbInfo[0];
            if (useM9330)
            {
#if M9330
                M9330 = new AgM933x();

                int errorcode = -1; string message = string.Empty;
                bool idquery = true;
                bool reset = true;
                // Open M9330A and load waveform
                string M9330options = string.Format("QueryInstrStatus=true, Simulate={0}, DriverSetup= DDS=false, Model=M9330A, Trace=false",
                    (simulateHardware ? "true" : "false"));

                //Initialize M9390
                M9330.Initialize(arbResource, idquery, reset, M9330options);
                // Clear startup messages & warnings if any.
                do
                {
                    M9330.Utility.ErrorQuery(ref errorcode, ref message);
                    if (errorcode != 0)
                        Console.WriteLine(message);
                } while (errorcode != 0);

                // Setup the M9330A for ET Waveform
                M9330.Arbitrary.Waveform.Predistortion.Enabled = false;
                M9330.Output.ReferenceClockSource = AgM933xReferenceClockSourceEnum.AgM933xReferenceClockSourceExternal;
                M9330.Output.ConfigureSampleClock(AgM933xSampleClockSourceEnum.AgM933xSampleClockSourceInternal, 1.25e9);
                M9330.Arbitrary.SampleRate = 1.25e9;
                M9330.Output.set_OperationMode("1", AgM933xOperationModeEnum.AgM933xOperationModeBurst);
                M9330.Output.OutputMode = AgM933xOutputModeEnum.AgM933xOutputModeArbitrary;
                M9330.Trigger.set_BurstCount("1", 1000000);
                M9330.Trigger.set_Source("1", AgM933xTriggerSourceEnum.AgM933xTriggerSourceExternal1);
                M9330.Trigger.ThresholdA = 0.75;// 1.0;
                M9330.Output.set_Configuration("1", AgM933xOutputConfigurationEnum.AgM933xOutputConfigurationAmplified);
                M9330.Output.set_FilterBandwidth("1", AgM933xFilterBandwidthEnum.AgM933xFilterBandwidth250MHz);
                M9330.Output.set_FilterEnabled("1", true);
                M9330.Output.set_Enabled("1", true);
                // Enable the sync output on all the markers
                M9330.Marker.ActiveMarker = "1";
                M9330.Marker.Source = AgM933xMarkerEnum.AgM933xMarkerWaveformRepeat;
                M9330.Marker.ActiveMarker = "2";
                M9330.Marker.Source = AgM933xMarkerEnum.AgM933xMarkerWaveformRepeat;
                M9330.Marker.ActiveMarker = "3";
                M9330.Marker.Source = AgM933xMarkerEnum.AgM933xMarkerWaveformRepeat;
                M9330.Marker.ActiveMarker = "4";
                M9330.Marker.Source = AgM933xMarkerEnum.AgM933xMarkerWaveformRepeat;
#endif
            }
            else if (use33522B)
            {
                if (oRm == null)
                    oRm = new ResourceManager();
                arbIo = new FormattedIO488();

                //Open session for instrument.
                arbIo.IO = (IMessage)oRm.Open(arbResource, AccessMode.NO_LOCK, 2000, "");
                arbIo.IO.Timeout = 10000;
                arbIo.WriteString("*RST");
                arbIo.WriteString("SOUR1:BURS:STAT OFF");
                arbIo.WriteString("TRIG1:SOUR EXT");
                arbIo.WriteString("TRIG1:DEL 0");
                arbIo.WriteString("TRIG1:SLOP POS");
                arbIo.WriteString(":ABORt");
                arbIo.WriteString("OUTP1 0");
                arbIo.WriteString("DATA:VOLatile:CLEar");

                arbIo.WriteString("SOUR1:BURS:NCYC INF");
                arbIo.WriteString("SOUR1:FUNC:ARB:FILT NORM");
                arbIo.WriteString("SOUR1:FUNC:ARB:ADV SRAT");
                arbIo.WriteString("SOUR1:BURS:MODE TRIG");
                arbIo.WriteString("SOUR1:BURS:STAT ON");
                string s;
                do
                {
                    arbIo.WriteString("SYSTEM:ERROR?");
                    s = arbIo.ReadString();
                } while (!s.ToUpper().Contains("NO ERROR"));
                //if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("33522B Error:\n" + s));

            }
        }

        public void closeEtArb()
        {
            // Close the ET ARB, if opened
#if M9330
            if (M9330 != null)
                if (M9330.Initialized)
                {
                    M9330.Close();
                    M9330 = null;
                }
#endif

        }

        public void clearAwgMemory()
        {
            if (!useEtArb)
                return;
            if (useM9330)
            {
#if M9330
                    M9330.Arbitrary.Waveform.set_Handle("1", etArbFiles.Last().handle);

#endif
            }
            else if (use33522B)
            {
                // Turn off outputs
                arbIo.WriteString("OUTP1 0");
                arbIo.WriteString("OUTP2 0");
                //Clear volatile memory
                arbIo.WriteString("SOURce1:DATA:VOLatile:CLEar", true);
                arbIo.WriteString("SOURce2:DATA:VOLatile:CLEar", true);
                //Read Errors
                arbIo.WriteString("SYSTEM:ERROR?", true);
                string s = arbIo.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("33522B Error:\n" + s));
            }
        }

        public void loadAwgWaveform(string filePath, string fileName, string refName, double vRef, double etpsGain, bool useShortWaveform = false, double shortTime = 0)
        {
            if (!useEtArb)
                return;
            // Add a new entry into the ArbInfo Array
            Array.Resize(ref etArbFiles, etArbFiles.Length + 1);
            etArbFiles[etArbFiles.Length - 1] = new ArbInfo();
            etArbFiles.Last().fileName = fileName;
            etArbFiles.Last().refName = refName;
            if (useM9330)
            {
#if M9330
                double[] waveformData = null;
                double arbClockRate = 1.25e9;
                double etSampleRate = 0;
                if (fileName.Contains(".bin"))
                {
                    // Setup to read the .bin files that are resampled from the ouptut of Signal Studio ET tool
                    var rs = new FileStream(filePath + fileName, FileMode.Open, FileAccess.Read);
                    BinaryReader rd = new BinaryReader(rs);
                    // Read the data size values
                    FileInfo fi = new FileInfo(filePath + fileName);
                    long waveformSize = fi.Length / 8 - 1;

                    long remainder = 0;
                    Math.DivRem(waveformSize, 8, out remainder);
                    if (remainder != 0)
                    {
                        waveformSize -= remainder;
                    }
                    arbClockRate = 1.25e9;
                    if (useShortWaveform)
                        waveformSize = (int)Math.Round(shortTime * arbClockRate);
                    waveformData = new double[waveformSize]; 

                    etSampleRate = rd.ReadDouble();
                    if (etSampleRate != arbClockRate)  throw (new Exception(fileName + " at incorrect Sample Rate: " + etSampleRate.ToString()));

                    for (int i = 0; i < waveformSize; i++)
                    {
                        waveformData[i] = rd.ReadDouble();
                    }
                    etArbFiles.Last().sampleRate = arbClockRate;
                    etArbFiles.Last().samples = waveformData.Length;
                    etArbFiles.Last().handle = M9330.Arbitrary.Waveform.Create(ref waveformData);
                    M9330.Arbitrary.Waveform.set_Handle("1", etArbFiles.Last().handle);
                }
#endif
            }
            else if (use33522B)
            {
                if (fileName.Contains(".csv"))
                {
                    // Setup to read the .CSV files that are output from the Signal Studio ET tool
                    var rs = new FileStream(fileName, FileMode.Open, FileAccess.Read);
                    TextReader rd = new StreamReader(rs);
                    // Read the data size values
                    string fileString = rd.ReadToEnd();
                    int lines = fileString.Count(x => x == '\n');
                    string[] waveformStrings = fileString.Split('\n');
                    float[] waveformData = new float[waveformStrings.Length - 4];
                    char[] seperators = new char[] { ',', '\r', '\t' };

                    double etSampleRate = 0.0;
                    int linesRead = 0;
                    int dataRead = 0;
                    while (dataRead < waveformData.Length && linesRead < waveformStrings.Length)
                    {
                        fileString = waveformStrings[linesRead];
                        fileString = fileString.ToUpper();
                        if (fileString.Contains("XDELTA"))
                        {
                            string[] columns = fileString.Split(seperators);
                            etSampleRate = (double)Math.Round((1 / Convert.ToDouble(columns[1])));
                        }
                        else if (fileString.Contains("UNIT") || fileString.Contains("Y,"))
                        {
                            // Skip these rows
                        }
                        else
                        {
                            // Read the ET data
                            string[] columns = fileString.Split(',');
                            waveformData[dataRead] = Convert.ToSingle(columns[0]);
                            dataRead++;
                        }
                        linesRead++;
                    }

                    if (useShortWaveform)
                        Array.Resize(ref waveformData, (int)Math.Round(shortTime * etSampleRate));

                    // Waveform Data is at ETPS Output
                    // Account for Reference Voltage and ETPS Gain
                    for (int i = 0; i < waveformData.Length; i++)
                    {
                        waveformData[i] = (waveformData[i] - (float)vRef) / (float)etpsGain;
                    }
                    float refFactor = (waveformData.Max() - waveformData.Min()) / (Math.Max(Math.Abs(waveformData.Max()), Math.Abs(waveformData.Min())) * 2);
                    for (int i = 0; i < waveformData.Length; i++)
                    {
                        waveformData[i] = waveformData[i] / refFactor;
                    }

                    short[] waveformDataInt = new short[waveformData.Length];
                    for (int i = 0; i < waveformData.Length; i++)
                    {
                        waveformDataInt[i] = (short)(waveformData[i] * 32767);
                    }
                    etArbFiles.Last().sampleRate = etSampleRate;
                    etArbFiles.Last().samples = waveformData.Length;
                    string s;
                    arbIo.IO.Timeout = 40000;
                    arbIo.WriteString("FORM:BORD NORM", true);
                    string cmd = "";
                    cmd = string.Format("SOURce1:DATA:ARB:DAC {0}, ", refName.Replace(" ", "_").Substring(0, 8));
                    arbIo.WriteIEEEBlock(cmd, waveformDataInt, true);
                    // Set timeout to a long time, waiting for file to load
                    arbIo.WriteString("*WAI");
                    //s = arbIo.ReadString();
                    arbIo.IO.Timeout = 10000;
                    cmd = string.Format("SOURCE1:FUNCtion:ARB:SRATe {0}", etSampleRate); // set sample rate
                    arbIo.WriteString(cmd);

                    //Read Errors
                    arbIo.WriteString("SYSTEM:ERROR?", true);
                    s = arbIo.ReadString();
                    if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("33522B Error:\n" + s));
                }
                else if (fileName.Contains(".BARB"))
                {
                    // Load .BARB file from 33522B NV Memory
                    arbIo.WriteString("MMEM:LOAD:DATA \"INT:\\" + fileName + "\"\n");
                    string s;
                    // Set timeout to a long time, waiting for file to load
                    arbIo.IO.Timeout = 40000;
                    arbIo.WriteString("*OPC?");
                    s = arbIo.ReadString();
                    arbIo.IO.Timeout = 10000;
                    arbIo.WriteString("SYSTEM:ERROR?\n");
                    s = arbIo.ReadString();
                    if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("33522B Error:\n" + s));
                }

            }
        }

        public void startAwgWaveformTriggered(string fileName, double gain, double offset, double triggerDelay, bool diff = false)
        {
            if (!useEtArb)
                return;
            ArbInfo currentArb = null;
            foreach (ArbInfo thisArb in etArbFiles)
            {
                if (thisArb.fileName.Contains(fileName))
                {
                    currentArb = thisArb;
                    break;
                }
            }
            if (currentArb == null) throw (new Exception("ET ARB Files not loaded:\n" + fileName));

            if (useM9330)
            {
#if M9330
                M9330.AbortGeneration();
                M9330.Arbitrary.set_Gain("1", gain);
                M9330.Arbitrary.set_Offset("1", offset);
                M9330.Arbitrary.Waveform.set_Handle("1", currentArb.handle);
                M9330.InitiateGeneration();
#endif
            }
            else if (use33522B)
            {
                string cmd;
                // check to see if this ARB is already playing.  Improtant for dual channel as it takes some time to copy the waveform
                // to the second channel.
                if (currentArb != lastArb)
                {
                    if (fileName.Contains(".csv"))
                    {
                        cmd = string.Format("SOURce1:FUNCtion:ARBitrary {0}, ", currentArb.refName.Replace(" ", "_").Substring(0, 8));
                        arbIo.WriteString(cmd);
                        cmd = string.Format("SOURCE1:FUNCtion:ARB:SRATe {0}", currentArb.sampleRate); // set sample rate
                        arbIo.WriteString(cmd);
                    }
                    else if (fileName.Contains(".BARB"))
                    {
                        arbIo.WriteString("SOUR1:FUNC:ARB \"INT:\\" + currentArb.refName + "\"\n");
                    }
                    lastArb = currentArb;
                    arbIo.WriteString("SOUR1:BURS:NCYC INF\n");
                    arbIo.WriteString("SOUR1:FUNC:ARB:FILT NORM\n");
                    arbIo.WriteString("SOUR1:FUNC:ARB:ADV SRAT\n");
                    arbIo.WriteString("SOUR1:BURS:MODE TRIG\n");
                    arbIo.WriteString("SOUR1:BURS:STAT ON\n");
                    arbIo.WriteString("SOUR1:FUNC ARB\n");
                    arbIo.WriteString("OUTP1 1\n");
                    if (diff)
                    {
                        arbIo.WriteString("SOUR1:TRACK INV\n");
                        arbIo.WriteString("OUTP2 1\n");
                    }
                }
                cmd = string.Format("SOUR1:VOLT {0}\n", gain);
                arbIo.WriteString(cmd);
                cmd = string.Format("SOUR1:VOLT:OFFS {0}\n", offset);
                arbIo.WriteString(cmd);
                arbIo.WriteString(":ABORt\n");
                arbIo.WriteString("SYSTEM:ERROR?\n");
                string s;
                s = arbIo.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("33522B Error:\n" + s));
            }

        }
        #region Create Waveforms

        // generateDcRampArb
        // Generate a DC Ramp signal for the ET AWG
        // Vary the amplitude based on min and max power fields
        public void generateDcRampArb(string refName, double minVoltage, double maxVoltage, double duration, double sampleRate)
        {
            if (!useEtArb)
                return;
            // Add a new entry into the ArbInfo Array  Treat the generated waveforms like the csv files
            Array.Resize(ref etArbFiles, etArbFiles.Length + 1);
            etArbFiles[etArbFiles.Length - 1] = new ArbInfo();
            etArbFiles.Last().fileName = refName;
            if (use33522B)
            {
                // Determine number of points
                int numPoints = (int)(duration * sampleRate);

                // Calcualte the scale factors
                double amplitude = maxVoltage;
                double scaleIncr = (maxVoltage - minVoltage) / (numPoints);

                // Create waveform and build array
                float[] waveform = new float[numPoints * 2];
                for (int i = 0; i < waveform.Length / 2; i++)
                {
                    // Create 
                    waveform[i] = (float)(minVoltage + i * scaleIncr);
                    waveform[waveform.Length - i - 1] = waveform[i];
                }
                etArbFiles.Last().sampleRate = sampleRate;
                etArbFiles.Last().samples = waveform.Length;
                arbIo.WriteString("FORM:BORD NORM", true);
                string cmd = string.Format("SOURce1:DATA:ARBitrary {0}, ", etArbFiles.Last().fileName.Replace(" ", "_").Substring(0, 8));
                arbIo.WriteIEEEBlock(cmd, waveform, true);
                // Set timeout to a long time, waiting for file to load
                arbIo.WriteString("*WAI");
                //s = arbIo.ReadString();
                arbIo.IO.Timeout = 10000;
                cmd = string.Format("SOURCE1:FUNCtion:ARB:SRATe {0}", sampleRate); // set sample rate
                arbIo.WriteString(cmd);
            }
        }

        // generateDcPulseArb
        // Generate a DC Pulse signal for the ET AWG
        // Vary the amplitude based on min and max power fields
        public void generateDcPulseArb(string refName, double pulseWidth, double duration, double sampleRate)
        {
            if (!useEtArb)
                return;
            // Add a new entry into the ArbInfo Array  Treat the generated waveforms like the csv files
            Array.Resize(ref etArbFiles, etArbFiles.Length + 1);
            etArbFiles[etArbFiles.Length - 1] = new ArbInfo();
            etArbFiles.Last().fileName = refName;
            if (use33522B)
            {
                // Determine number of points
                int numPoints = (int)(duration * sampleRate);
                int pulsePoints = (int)(pulseWidth * sampleRate);

                // Calcualte the scale factors
                //double amplitude = maxVoltage;
                //double scaleIncr = (maxVoltage - minVoltage) / (numPoints);

                // Create waveform and build array
                float[] waveform = new float[numPoints];
                for (int i = 0; i < pulsePoints; i++)
                {
                    // Create 
                    waveform[i] = 1.0f;
                }
                for (int i = pulsePoints; i < waveform.Length; i++)
                {
                    // Create 
                    waveform[i] = 0.0f;
                }
                etArbFiles.Last().sampleRate = sampleRate;
                etArbFiles.Last().samples = waveform.Length;
                arbIo.WriteString("FORM:BORD NORM", true);
                string cmd = string.Format("SOURce1:DATA:ARBitrary {0}, ", etArbFiles.Last().fileName.Replace(" ", "_").Substring(0, 8));
                arbIo.WriteIEEEBlock(cmd, waveform, true);
                // Set timeout to a long time, waiting for file to load
                arbIo.WriteString("*WAI");
                //s = arbIo.ReadString();
                arbIo.IO.Timeout = 10000;
                cmd = string.Format("SOURCE1:FUNCtion:ARB:SRATe {0}", sampleRate); // set sample rate
                arbIo.WriteString(cmd);
            }
        }
        // playPulseMode
        // Generate a DC Pulse signal for the ET AWG
        // Using the 33522B Pulse Mode
        public void playPulseMode(string refName, double leadTime, double lagTime, double dutyCycle, double pulseVoltage, double triggerDelay)
        {
            if (!useEtArb)
                return;


            //// Get Sample Rate of ARB File
            //M9381.Modulation.IQ.ArbInformation(refName, ref arbSampleRate, ref arbRmsValue, arbScale, rfMarker, alcMarker);

            //// Deterimne Pulse Width and duty cycle
            //// Trailing time = (PulseWidth/DC) - PulseWidth - LeadTime
            //// WLAN Waveforms are 100% duty Cycle, Get Pulse Width from number of samples
            //double rfPulsewidth = getNumberSamples(refName) / arbSampleRate;
            //double dcPulsewidth = rfPulsewidth + leadTime + lagTime;
            //double period = rfPulsewidth / (1 - dutyCycle);

            //string cmd;
            //arbIo.WriteString("SOUR1:BURS:NCYC 1\n");
            //arbIo.WriteString("SOUR1:BURS:MODE TRIG\n");
            //arbIo.WriteString("SOUR1:BURS:STAT ON\n");
            //arbIo.WriteString("SOUR1:FUNC PULSE\n");
            //cmd = string.Format("SOUR1:FUNCtion:PULSe:WIDTh {0}\n", dcPulsewidth);
            //arbIo.WriteString(cmd);
            //cmd = string.Format("SOUR1:FUNCtion:PULSe:PERiod {0}\n", period * .99);
            //arbIo.WriteString(cmd);
            //cmd = string.Format("SOUR1:VOLT {0}\n", pulseVoltage);
            //arbIo.WriteString(cmd);
            //cmd = string.Format("SOUR1:VOLT:OFFS {0}\n", pulseVoltage / 2);
            //arbIo.WriteString(cmd);
            //cmd = string.Format("TRIG1:DEL {0}\n", triggerDelay);
            //arbIo.WriteString(cmd);
            //arbIo.WriteString("OUTP1 1\n");
            //arbIo.WriteString(":ABORt\n");
            //arbIo.WriteString("SYSTEM:ERROR?\n");
            //string s;
            //s = arbIo.ReadString();
            //if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("33522B Error:\n" + s));
        }

        #endregion
    }
    public enum arbType
    {
        CSV = 0,
        BIN = 1,
        BARB = 2,
    }
    public class ArbInfo
    {
        public string fileName { set; get; }
        public string refName { set; get; }
        public int handle { set; get; }
        public int samples { set; get; }
        public double sampleRate { set; get; }
        public double amplitude { set; get; }
        public double offset { set; get; }
        public ArbInfo()
        {
            refName = "";
            fileName = "";
            handle = 0;
            sampleRate = 1;
            samples = 0;
            amplitude = 0.0;
            offset = 0.0;
        }
    }
}
